﻿#region << Using Directives >>
using System;
#endregion

namespace Volpe.Cafe.Generic
{
    /// <summary>
    /// Represents an object that stores compliance modeling data for each fuel type.
    /// </summary>
    [Serializable]
    public class FTValue<T> : ICloneable
    {

        #region /*** Ctors ***/

        /// <summary>
        /// Initializes a new instance of the <see cref="FTValue{T}"/> class.
        /// </summary>
        public FTValue()
        {
            this.Items = new T[Classes.Length];
        }
        /// <summary>
        /// Initializes a new instance of the <see cref="FTValue{T}"/> class using the specified values.
        /// </summary>
        public FTValue(T gasoline, T ethanol85, T diesel, T electricity, T hydrogen, T cng)
        {
            this.Items = new T[] { gasoline, ethanol85, diesel, electricity, hydrogen, cng };
        }

        #endregion

        #region /*** Methods ***/

        #region /* Overloaded Operators */

        /// <summary>
        /// Determines whether the specified <see cref="FTValue{T}"/> values are equal.
        /// </summary>
        /// <param name="value1">The first value to compare.</param>
        /// <param name="value2">The second value to compare.</param>
        /// <returns>true, if the two <see cref="FTValue{T}"/> values are equal; false, otherwise.</returns>
        public static bool operator ==(FTValue<T> value1, FTValue<T> value2)
        {
            return Equals(value1, value2);
        }
        /// <summary>
        /// Determines whether the specified <see cref="FTValue{T}"/> values are not equal.
        /// </summary>
        /// <param name="value1">The first value to compare.</param>
        /// <param name="value2">The second value to compare.</param>
        /// <returns>true, if the two <see cref="FTValue{T}"/> values are not equal; false, otherwise.</returns>
        public static bool operator !=(FTValue<T> value1, FTValue<T> value2)
        {
            return !Equals(value1, value2);
        }

        #endregion

        #region /* ICloneable Members */

        /// <summary>
        /// Creates a shallow copy of the current <see cref="FTValue{T}"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="FTValue{T}"/>.</returns>
        object ICloneable.Clone()
        {
            return this.Clone();
        }
        /// <summary>
        /// Creates a shallow copy of the current <see cref="FTValue{T}"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="FTValue{T}"/>.</returns>
        public FTValue<T> Clone()
        {
            FTValue<T> value = new FTValue<T>();
            this.CopyTo(value);
            return value;
        }
        /// <summary>
        /// Copies the members of the current <see cref="FTValue{T}"/> instance into the specified value.
        /// </summary>
        /// <param name="value">The value where to copy the members of the current instance.</param>
        protected void CopyTo(FTValue<T> value)
        {
            for (int i = 0; i < this.Items.Length; i++)
            {
                value.Items[i] = this.Items[i];
            }
        }

        #endregion

        #region /* Overriden from object */

        /// <summary>
        /// Returns the string representation of this <see cref="FTValue{T}"/> instance.
        /// </summary>
        /// <returns>The string representation of the <see cref="FTValue{T}"/> instance.</returns>
        public override string ToString()
        {
            string s = string.Empty;
            for (int i = 0; i < this.Items.Length; i++)
            {
                if (i > 0) { s += ", "; }
                s += (Names[i] + "=" + this.Items[i].ToString());
            }
            return "{" + s + "}";
        }

        /// <summary>
        /// Serves as a hash function for a particular type, suitable for use in hashing algorithms and data structures like a hash table.
        /// </summary>
        /// <returns>A hash code for the current <see cref="FTValue{T}"/>.</returns>
        public override int GetHashCode()
        {
            int hash = 0;
            for (int i = 0; i < this.Items.Length; i++)
            {
                hash = hash ^ this.Items[i].GetHashCode();
            }
            return hash;
        }

        /// <summary>
        /// Determines whether the specified <see cref="Object"/> is equal to the current <see cref="FTValue{T}"/> value.
        /// </summary>
        /// <param name="obj">The <see cref="Object"/> to compare with the current <see cref="FTValue{T}"/> value.</param>
        /// <returns>true, if the specified <see cref="Object"/> is equal to the current <see cref="FTValue{T}"/> value; false, otherwise.</returns>
        public override bool Equals(object obj)
        {
            return (obj is FTValue<T>) ? this.Equals((FTValue<T>)obj) : base.Equals(obj);
        }
        /// <summary>
        /// Determines whether the specified <see cref="FTValue{T}"/> value is equal to the current <see cref="FTValue{T}"/> value.
        /// </summary>
        /// <param name="value">The <see cref="FTValue{T}"/> value to compare with the current <see cref="FTValue{T}"/> value.</param>
        /// <returns>true, if the specified <see cref="FTValue{T}"/> value is equal to the current <see cref="FTValue{T}"/> value;
        ///   false, otherwise.</returns>
        public bool Equals(FTValue<T> value)
        {
            return Equals(this, value);
        }
        /// <summary>
        /// Determines whether the specified <see cref="FTValue{T}"/> values are equal.
        /// </summary>
        /// <param name="value1">The first <see cref="FTValue{T}"/> value to compare.</param>
        /// <param name="value2">The second <see cref="FTValue{T}"/> value to compare.</param>
        /// <returns>true, if the specified <see cref="FTValue{T}"/> values are equal; false, otherwise.</returns>
        public static bool Equals(FTValue<T> value1, FTValue<T> value2)
        {
            if (object.ReferenceEquals(value1, null) && object.ReferenceEquals(value2, null)) { return true ; }
            if (object.ReferenceEquals(value1, null) || object.ReferenceEquals(value2, null)) { return false; }
            //
            for (int i = 0; i < value1.Items.Length; i++)
            {
                if (!value1.Items[i].Equals(value2.Items[i])) { return false; }
            }
            return true;
        }

        #endregion

        /// <summary>
        /// Resets all members of this <see cref="FTValue{T}"/> instance to their default values.
        /// </summary>
        public void Clear()
        {
            for (int i = 0; i < this.Items.Length; i++)
            {
                this.Items[i] = default(T);
            }
        }

        /// <summary>
        /// Returns the index corresponding to the specified <see cref="FuelType"/>.
        /// </summary>
        /// <param name="fuelType"></param>
        /// <returns>The index corresponding to the specified <see cref="FuelType"/>.</returns>
        protected int GetIndex(FuelType fuelType)
        {
            if      (fuelType == FuelType.Gasoline   ) { return  0; }
            else if (fuelType == FuelType.Ethanol85  ) { return  1; }
            else if (fuelType == FuelType.Diesel     ) { return  2; }
            else if (fuelType == FuelType.Electricity) { return  3; }
            else if (fuelType == FuelType.Hydrogen   ) { return  4; }
            else if (fuelType == FuelType.CNG        ) { return  5; }
            else                                       { return -1; }
        }

        #endregion

        #region /*** Properties ***/

        /// <summary>Gets or sets a fuel value component for the specified <see cref="FuelType"/>.</summary>
        /// <exception cref="System.IndexOutOfRangeException">The value specified does not represent a valid fuel type.</exception>
        public T this[FuelType fuelType]
        {
            get { return this.Items[this.GetIndex(fuelType)]; }
            set { this.Items[this.GetIndex(fuelType)] = value; }
        }

        /// <summary>Gets an array of compliance modeling data specified by <see cref="FuelType"/>.</summary>
        public T[] Items { get; private set; }

        ///// <summary>Gets or sets the compliance modeling data for Gasoline fuel.</summary>
        //public T Gasoline { get; set; }
        ///// <summary>Gets or sets the compliance modeling data for E85 fuel (blend of 85% Ethanol, 15% Gasoline).</summary>
        //public T Ethanol85 { get; set; }
        ///// <summary>Gets or sets the compliance modeling data for Diesel fuel.</summary>
        //public T Diesel { get; set; }
        ///// <summary>Gets or sets the compliance modeling data for B20 fuel (blend of 20% Biodiesel, 80% Petrodiesel).</summary>
        //public T Biodiesel20 { get; set; }
        ///// <summary>Gets or sets the compliance modeling data for Electricity fuel type.</summary>
        //public T Electricity { get; set; }
        ///// <summary>Gets or sets the compliance modeling data for Hydrogen fuel type.</summary>
        //public T Hydrogen { get; set; }
        ///// <summary>Gets or sets the compliance modeling data for Compressed Natural Gas fuel.</summary>
        //public T CNG { get; set; }
        ///// <summary>Gets or sets the compliance modeling data for Liquefied Natural Gas fuel.</summary>
        //public T LNG { get; set; }
        ///// <summary>Gets or sets the compliance modeling data for Liquefied Petroleum Gas fuel.</summary>
        //public T LPG { get; set; }

        /// <summary>Gets whether this <see cref="FTValue{T}"/> instance has actual (non-default) values for more then one component.</summary>
        public bool IsMultiFuel
        {
            get
            {
                int f = (int)this.FuelType;
                return ((f & (f - 1)) != 0);
            }
        }
        /// <summary>Gets the fuel type that represents all componenets of this <see cref="FTValue{T}"/> instance.</summary>
        public FuelType FuelType
        {
            get
            {
                FuelType fuelType = FuelType.None;
                for (int i = 0; i < this.Items.Length; i++)
                {
                    if (this.Items[i] != null && !this.Items[i].Equals(default(T))) { fuelType |= Classes[i]; }
                }
                return fuelType;
            }
        }

        #endregion

        #region /*** Variables ***/

        /// <summary>Specifies the <see cref="FuelType"/>s supported by this object. This field is read-only.</summary>
        public static readonly FuelType[] Classes  = new FuelType[] { FuelType.Gasoline,
                                                                      FuelType.Ethanol85,
                                                                      FuelType.Diesel,
                                                                      FuelType.Electricity,
                                                                      FuelType.Hydrogen,
                                                                      FuelType.CNG };
        /// <summary>Provides the name for each of the corresponing <see cref="FuelType"/> <see cref="Classes"/>. This field is read-only.</summary>
        public static readonly string  [] Names    = new string  [] { "Gasoline",
                                                                      "Ethanol-85%",
                                                                      "Diesel",
                                                                      "Electricity",
                                                                      "Hydrogen",
                                                                      "Compressed Natural Gas" };
        /// <summary>Provides an acronym (useful for reporting) for each of the corresponing <see cref="FuelType"/> <see cref="Classes"/>. This field is read-only.</summary>
        public static readonly string  [] Acronyms = new string  [] { "G",
                                                                      "E85",
                                                                      "D",
                                                                      "E",
                                                                      "H",
                                                                      "CNG" };

        #endregion

    }
}
